package gov.va.vamf.scheduling.direct.eligibilitychecker

import gov.va.vamf.scheduling.direct.datalayer.eligibility.DirectBookingEligibilityCriteriaService
import gov.va.vamf.scheduling.direct.datalayer.eligibility.RequestEligibilityCriteriaService
import gov.va.vamf.scheduling.direct.domain.CoreSetting
import gov.va.vamf.scheduling.direct.domain.CoreSettings
import gov.va.vamf.scheduling.direct.domain.CustomRequestSetting
import gov.va.vamf.scheduling.direct.domain.CustomRequestSettings
import gov.va.vamf.scheduling.direct.domain.DirectBookingEligibilityCriteria
import gov.va.vamf.scheduling.direct.domain.RequestEligibilityCriteria
import spock.lang.Specification

class EligibilityCheckWorkerSpec extends Specification {
    private EligibilityCheckWorker eligibilityCheckWorker
    private DirectBookingEligibilityCriteriaService directCriteriaStub
    private RequestEligibilityCriteriaService requestCriteriaStub

    private static final STATION_CODE = '523'
    private static final TYPE_OF_CARE = '203'

    def setup() {
        directCriteriaStub = Stub(DirectBookingEligibilityCriteriaService)
        requestCriteriaStub = Stub(RequestEligibilityCriteriaService)
        eligibilityCheckWorker = new EligibilityCheckWorker(null, null, requestCriteriaStub, null, directCriteriaStub)
    }

    def 'EligibilityCheckWorker#getDirectHistoryDurationInDays(String, String) should return 0 if the patient history setting is not found'() {
        given:
        directCriteriaStub.retrieveDirectBookingEligibilityCriteria(_) >> criteria

        expect:
        eligibilityCheckWorker.getDirectHistoryDurationInDays(TYPE_OF_CARE, STATION_CODE) == 0

        where:
        criteria << [
                // no configuration at all for the station
                null,
                // no settings for the station
                new DirectBookingEligibilityCriteria(),
                // settings exist for the type of care, but patient history is null
                makeDirectCriteria([new CoreSetting(TYPE_OF_CARE, '', null, null)] as CoreSettings)
        ]
    }

    def 'EligibilityCheckWorker#getDirectHistoryDurationInDays(String, String) should return the patient history setting when it is found'() {
        given:
        def criteria = makeDirectCriteria([new CoreSetting(TYPE_OF_CARE, '', null, 1)] as CoreSettings)
        directCriteriaStub.retrieveDirectBookingEligibilityCriteria(_) >> criteria

        expect:
        eligibilityCheckWorker.getDirectHistoryDurationInDays(TYPE_OF_CARE, STATION_CODE) == 1
    }

    def 'EligibilityCheckWorker#getRequestHistoryDurationInDays(String, String) should return 0 if the patient history setting is not found'() {
        given:
        requestCriteriaStub.retrieveRequestEligibilityCriteria(_) >> criteria

        expect:
        eligibilityCheckWorker.getRequestHistoryDurationInDays(TYPE_OF_CARE, STATION_CODE) == 0

        where:
        criteria << [
                // no configuration at all for the station
                null,
                // no settings for the station
                new RequestEligibilityCriteria(),
                // settings exist for the type of care, but patient history is null
                makeRequestCriteria(
                        [new CoreSetting(TYPE_OF_CARE, '', null, null)] as CoreSettings,
                        [new CustomRequestSetting(TYPE_OF_CARE, '', null, null)] as CustomRequestSettings
                )
        ]
    }

    def 'EligibilityCheckWorker#getRequestHistoryDurationInDays(String, String) should return the patient history setting when it is found'() {
        given:
        def criteria = makeRequestCriteria(
                [new CoreSetting(TYPE_OF_CARE, '', null, 1)] as CoreSettings,
                [new CustomRequestSetting('CR1', '', null, null)] as CustomRequestSettings
        )
        requestCriteriaStub.retrieveRequestEligibilityCriteria(_) >> criteria

        expect:
        eligibilityCheckWorker.getRequestHistoryDurationInDays(TYPE_OF_CARE, STATION_CODE) == 1
    }

    private def makeDirectCriteria(coreSettings) {
        [coreSettings: coreSettings] as DirectBookingEligibilityCriteria
    }

    private def makeRequestCriteria(requestSettings, customSettings) {
        [requestSettings: requestSettings, customRequestSettings: customSettings] as RequestEligibilityCriteria
    }

}
